<?php

// read configuration from file
function sfwm_settings()
{
    $config_file = plugin_dir_path(__FILE__).'settings.php';
    if ( file_exists($config_file) )
    {
        $config_data = get_file_data($config_file,array(
            'Table numbers'=>'Table numbers',
            'Page title format'=>'Page title format',
            'Page title suffix'=>'Page title suffix',
            'Page meta description'=>'Page meta description',
            'Meta description type'=>'Meta description type',
            'Meta description prefix'=>'Meta description prefix',
            'Table ID pad'=>'Table number padding',
            'Row ID pad'=>'Row number padding',
            'Page path'=>'Page path',
            'Slug'=>'Slug',
            'Slug format'=>'Slug format',
            'Slug preservation'=>'Slug preservation',
            'Slug replacement input'=>'Slug replacement input',
            'Slug replacement output'=>'Slug replacement output',
            'Sitemap'=>'Sitemap',
            'Sitemap name'=>'Sitemap name',
            'Validate settings'=>'Validate settings',
            'URL forward type'=>'URL forward type',
            'Admin notices'=>'Admin notices',
            'Configured'=>'Configured'
        ));
        define('SFWM_CFG',array_map('trim',$config_data));
    }
    else
    {
        define('SFWM_CFG',array(
            'Admin notices'=>'1',
            'Configured'=>'0'
        ));
    }
    if ( SFWM_CFG["Validate settings"] ) sfwm_validate_cfg();
}

// validate configuration
function sfwm_validate_cfg()
{
    if ( preg_match('/[^0-9,]|^$/',SFWM_CFG["Table numbers"]) )
    {
        set_transient('sfwm_cfg_error','Table numbers',10);
    }
    if ( preg_match('/[^a-zA-Z0-9 _-]|^$/',SFWM_CFG["Page title format"]) )
    {
        set_transient('sfwm_cfg_error','Page title format',10);
    }
    if ( !in_array(SFWM_CFG["Page meta description"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Page meta description',10);
    }
    if ( !in_array(SFWM_CFG["Meta description type"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Meta description type',10);
    }
    if ( preg_match('/[^0-9]/',SFWM_CFG["Table ID pad"]) )
    {
        set_transient('sfwm_cfg_error','Table number padding',10);
    }
    if ( preg_match('/[^0-9]/',SFWM_CFG["Row ID pad"]) )
    {
        set_transient('sfwm_cfg_error','Row number padding',10);
    }
    if ( !preg_match('/^[a-zA-Z0-9_-]+\/$|^$/',SFWM_CFG["Page path"]) )
    {
        set_transient('sfwm_cfg_error','Page path',10);
    }
    if ( !in_array(SFWM_CFG["Slug"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Slug',10);
    }
    if ( preg_match('/[^a-zA-Z0-9 _-]|^$/',SFWM_CFG["Slug format"]) )
    {
        set_transient('sfwm_cfg_error','Slug format',10);
    }
    if ( !in_array(SFWM_CFG["Slug preservation"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Slug preservation',10);
    }
    if ( preg_match('/[^a-zA-Z0-9,_-]/',SFWM_CFG["Slug replacement output"]) )
    {
        set_transient('sfwm_cfg_error','Slug replacement output',10);
    }
    if ( !in_array(SFWM_CFG["Sitemap"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Sitemap',10);
    }
    if ( preg_match('/[^a-zA-Z0-9_-]|^$/',SFWM_CFG["Sitemap name"]) )
    {
        set_transient('sfwm_cfg_error','Sitemap name',10);
    }
    if ( !in_array(SFWM_CFG["Validate settings"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Validate settings',10);
    }
    if ( !in_array(SFWM_CFG["URL forward type"], ['301','302']) )
    {
        set_transient('sfwm_cfg_error','URL forward type',10);
    }
    if ( !in_array(SFWM_CFG["Admin notices"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Admin notices',10);
    }
    if ( !in_array(SFWM_CFG["Configured"], ['0','1']) )
    {
        set_transient('sfwm_cfg_error','Configured',10);
    }
}

// run on plugin activation
function sfwm_activate()
{
    // check for required Master-Detail version
    $md_path = 'wdt-master-detail/wdt-master-detail.php';
    $md_min = '2.0'; 
    $md_error = 0;
    if ( file_exists(WP_PLUGIN_DIR.'/'.$md_path) )
    {
        $md_data = get_plugin_data(WP_PLUGIN_DIR.'/'.$md_path);
        $md_error = version_compare($md_data['Version'],$md_min,'>=') ? 0 : 1;
    }
    else
    {
        $md_error = 1;
    }
    if ( $md_error )
    {
        set_transient('sfwm_activation_error_md_old',true,10);
    }
    // check for plugin configuration
    $config_file = plugin_dir_path(__FILE__).'settings.php';
    $config_error = 0;
    if ( file_exists($config_file) )
    {
        $config_data = get_file_data($config_file,array(
            'Configured'=>'Configured'
        ));
        $config_error = (int) $config_data['Configured'] ? 0 : 1;
    }
    else
    {
        $config_error = 1;
    }
    if ( $config_error )
    {
        set_transient('sfwm_activation_error_not_configured',true,10);
    }
    // force rewrite rules to be recreated at the right time after activation
    delete_option('rewrite_rules');
}

// run on plugin deactivation
function sfwm_deactivate()
{
    // force rewrite rules to be recreated at the right time after deactivation
    delete_option('rewrite_rules');
}

// fetch template page name
function sfwm_tpl($tid)
{
    global $wpdb;
    $table = $wpdb->prefix.'wpdatatables';
    $tid = (int) $tid;
    $md_tpl = $wpdb->get_var("SELECT
        advanced_settings ->> '$.masterDetailRenderPage' md_tpl
        FROM $table WHERE id=$tid");
    $tpl_pagename = array_reverse(explode('/',rtrim($md_tpl,'/')))[0];
    return $tpl_pagename;
}

// fetch row ID column
function sfwm_id_col($tid)
{
    global $wpdb;
    $table = $wpdb->prefix.'wpdatatables_columns';
    $tid = (int) $tid;
    $id_col = $wpdb->get_var("SELECT orig_header
        FROM $table WHERE id_column=1 AND table_id=$tid");
    if ( !$id_col )
    {
        $id_col = 'wdt_ID'; // on error, defaulting to wdt_ID
        set_transient('sfwm_data_error_id_col',true,10);
    }
    return $id_col;
}

// process table ID number
function sfwm_table_id($tid)
{
    $tid = ltrim($tid,'0');
    if ( SFWM_CFG["Table ID pad"] )
    {
        $tid = str_pad($tid,SFWM_CFG["Table ID pad"],'0',STR_PAD_LEFT);
    }
    if ( !(int) $tid )
    {
        $tid = strtok(SFWM_CFG["Table numbers"],',');
    }
    return $tid;
}

// process row ID number
function sfwm_id($id)
{
    $id = ltrim($id,'0');
    if ( SFWM_CFG["Row ID pad"] )
    {
        $id = str_pad($id,SFWM_CFG["Row ID pad"],'0',STR_PAD_LEFT);
    }
    return $id;
}

// set mode (single table or several tables enabled)
function sfwm_set_mode($tid)
{
    $tid = ( !str_contains(SFWM_CFG["Table numbers"],',') ) ? '' : $tid.'-';
    return $tid;
}

// check URL against our clean URL format
function sfwm_match(&$matched)
{
    $url = $_SERVER["REQUEST_URI"];
    $result = preg_match('/^\/'.str_replace('/','\/',SFWM_CFG["Page path"])
        .'([0-9]+-)?([0-9]+)(-[a-zA-Z0-9.-]+)?\/?(?:\?.*)?$/i',
        $url, $matches);
    $matched = $matches;
    return $result;
}

// redirect to clean URL
function sfwm_redirect($item)
{
    if ( !$item )
    {
        $target_url = home_url();
    }
    else
    {
        $target_url = home_url().'/'.SFWM_CFG["Page path"].$item;
    }
    wp_redirect($target_url, SFWM_CFG["URL forward type"]);
    exit();
}

// generate sitemap
function sfwm_sitemap()
{
    global $wpdb;
    $table = $wpdb->prefix.'wpdatatables';
    $array = sfwm_table_array();
    foreach($array as $t)
    {
        $col_name = sfwm_id_col($t);
        $table_query = $wpdb->get_var("SELECT content AS table_query
            FROM $table WHERE id=$t");
        $ids = $wpdb->get_col("SELECT $col_name AS ids
            FROM ($table_query) t ORDER BY 1 DESC");
        foreach ($ids as $i)
        {
            $slug = ( SFWM_CFG["Slug"] ) ? '-'.sfwm_slug($t, $i) : '';
            $tid = str_pad($t,SFWM_CFG["Table ID pad"],'0',STR_PAD_LEFT);
            $tid = sfwm_set_mode($tid);
            $id = str_pad($i,SFWM_CFG["Row ID pad"],'0',STR_PAD_LEFT);
            $item = $tid.$id.$slug;
            $target_url = home_url().'/'.SFWM_CFG["Page path"].$item;
            echo $target_url."\n";
        }
    }
    exit();
}

// fetch data for title or slug
function sfwm_cols($tid, $id, $use)
{
    global $wpdb;
    $tid = (int) $tid;
    $id = (int) $id;
    if ( $use == 1 )
    {
        $cols = SFWM_CFG["Page title format"];
    }
    else if ( $use == 2 )
    {
        $cols = SFWM_CFG["Slug format"];
    }
    else
    {
        $cols = sfwm_id_col($tid);
    }
    $cols = preg_replace('/[^a-zA-Z0-9 _-]+/','',$cols);
    $cols = "IFNULL(`".str_replace(" ","`,''),' ',IFNULL(`",$cols)."`,'')";
    $table = $wpdb->prefix.'wpdatatables';
    $table_query = $wpdb->get_var("SELECT content AS table_query
        FROM $table WHERE id=$tid");
    $col_name = sfwm_id_col($tid);
    $cols_data = $wpdb->get_var("SELECT CONCAT($cols) AS cols_data
        FROM ($table_query) t WHERE $col_name=$id");
    if ( !$cols_data )
    {
        set_transient('sfwm_data_error_cols',true,10);
    }
    return $cols_data;
}

// construct title for page
function sfwm_title($tid, $id)
{
    $title = sfwm_cols($tid, $id, 1);
    return $title;
}

// construct slug for page
function sfwm_slug($tid, $id)
{
    $slug = sfwm_cols($tid, $id, 2);
    $orig_value = SFWM_CFG["Slug replacement input"];
    $orig_value = str_getcsv($orig_value);
    $replacement = SFWM_CFG["Slug replacement output"];
    $replacement = str_getcsv($replacement);
    $slug = str_replace($orig_value,$replacement,$slug);
    $slug = preg_replace('/[^a-zA-Z0-9.-]+/','-',$slug);
    if ( !SFWM_CFG["Slug preservation"] ) $slug = strtolower($slug);
    return $slug;
}

// convert table list to array
function sfwm_table_array()
{
    $list = SFWM_CFG["Table numbers"];
    $list = preg_replace('/[^,0-9]+/','',$list);
    $array = str_getcsv($list);
    return $array;
}

// register custom query vars
function sfwm_query_vars($query_vars)
{
    $query_vars[] = 'wdt_md_p_t_id';
    $query_vars[] = 'wdt_md_p_t_col_name';
    $query_vars[] = 'wdt_md_col_value';
    $query_vars[] = 'sfwm_sitemap';
    return $query_vars;
}

// set page title according to page data
function sfwm_set_title($title)
{
    $matches = array();
    if ( sfwm_match($matches) )
    {
        $tid = rtrim($matches[1],'-');
        if ( !(int) $tid )
        {
            $tid = strtok(SFWM_CFG["Table numbers"],',');
        }
        $id = $matches[2];
        $title = sfwm_title($tid, $id).' '.SFWM_CFG["Page title suffix"];
    }
    return $title;
}

// set page canonical to actual clean URL
function sfwm_set_canonical($canonical_url)
{
    $matches = array();
    if ( sfwm_match($matches) )
    {
        $canonical_url = home_url().$_SERVER["REQUEST_URI"];
    } 
    return $canonical_url;
}

// set sitemap content type to plain text
function sfwm_headers($headers)
{
    if ( get_query_var('sfwm_sitemap') )
    {
        $headers['Content-Type'] = 'text/plain; charset=UTF-8';
    }
    return $headers;     
}

// optimize headers
function sfwm_head()
{
    $matches = array();
    if ( sfwm_match($matches) )
    {
        // set page meta description
        if ( SFWM_CFG["Page meta description"] )
        {
            $tid = sfwm_table_id(rtrim($matches[1],'-'));
            $id = sfwm_id($matches[2]);
            if ( SFWM_CFG["Meta description type"] )
            {
                $tid = sfwm_set_mode($tid);
                $dt = SFWM_CFG["Meta description prefix"].' '.$tid.$id.'.';
            }
            else
            {
                $title = sfwm_title($tid, $id);
                $dt = SFWM_CFG["Meta description prefix"].' '.$title.'.';
            }
            echo '<meta name="description" content="'.esc_attr($dt).'" />'."\n";
        }
        // remove unnecessary meta tag (and HTTP header field) for shortlink
        remove_action('wp_head', 'wp_shortlink_wp_head', 10);
        remove_action('template_redirect', 'wp_shortlink_header', 11);
    }
}

// handle requested URLs, redirect as needed
function sfwm_request_handler()
{
    // generate sitemap
    if ( get_query_var('sfwm_sitemap') ) sfwm_sitemap();
    // redirect ugly URLs
    if ( isset($_GET['wdt_md_col_value']) )
    {
        $tid = (int) $_GET['wdt_md_p_t_id'];
        if ( in_array($tid,sfwm_table_array()) )
        {
            $tid = str_pad($tid,SFWM_CFG["Table ID pad"],'0',STR_PAD_LEFT).'-';
            $id = (int) $_GET['wdt_md_col_value'];
            $id = str_pad($id,SFWM_CFG["Row ID pad"],'0',STR_PAD_LEFT);
            $item = $tid.$id;
            sfwm_redirect($item);
        }
    }
    // catch clean URLs
    $matches = array();
    if ( sfwm_match($matches) )
    {
        $tid = sfwm_table_id(rtrim($matches[1],'-'));
        $id = sfwm_id($matches[2]);
        sfwm_cols($tid, $id, 0);
        if ( get_transient('sfwm_data_error_cols') )
        {
            delete_transient('sfwm_data_error_cols');
            $item = '';
            sfwm_redirect($item);
        }
        $slug = ( SFWM_CFG["Slug"] ) ? '-'.sfwm_slug($tid, $id) : '';
        $tid = sfwm_set_mode($tid);
        // ensure proper clean URLs, discard any query strings
        if ( $matches[0] != strtok($matches[0],'?')
            || $matches[1] !== $tid
            || $matches[2] !== $id
            || $matches[3] != $slug
        )
        {
            $item = $tid.$id.$slug;
            sfwm_redirect($item);
        }
    }
}

// set up rewrite rules
function sfwm_rewrite_rules()
{
    // ugly URLs to clean (short or friendly) URLs
    if ( str_contains(SFWM_CFG["Table numbers"],',') ) // multimode
    {
        $array = sfwm_table_array();
        foreach($array as $tid)
        {
            $tpl = sfwm_tpl($tid);
            $id_col = sfwm_id_col($tid);
            add_rewrite_rule(
                '^'.SFWM_CFG["Page path"]
                .'0*'.$tid.'-0*([0-9]+)(?:-[a-zA-Z0-9.-]+)?/?$',
                'index.php?pagename='.$tpl
                .'&wdt_md_p_t_id='.$tid
                .'&wdt_md_p_t_col_name='.$id_col
                .'&wdt_md_col_value=$matches[1]',
                'top'
            );
        }
    }
    $tid = strtok(SFWM_CFG["Table numbers"],','); // use first ID, none in URL
    $tpl = sfwm_tpl($tid);
    $id_col = sfwm_id_col($tid);
    add_rewrite_rule(
        '^'.SFWM_CFG["Page path"]
        .'0*([0-9]+)(?:-[a-zA-Z0-9.-]+)?/?$',
        'index.php?pagename='.$tpl
        .'&wdt_md_p_t_id='.$tid
        .'&wdt_md_p_t_col_name='.$id_col
        .'&wdt_md_col_value=$matches[1]',
        'top'
    );
    // dynamic sitemap
    if ( SFWM_CFG["Sitemap"] )
    {
        add_rewrite_rule(
            '^'.SFWM_CFG["Sitemap name"].'\.txt$',
            'index.php?sfwm_sitemap=1',
            'top'
        );
    }
}

// shamelessly modify super global so that Master-Detail gets the data it wants
function sfwm_pass_request()
{
    if ( get_query_var('wdt_md_p_t_id') )
    {
        $_REQUEST['wdt_md_p_t_id'] = get_query_var('wdt_md_p_t_id');
        $_REQUEST['wdt_md_p_t_col_name'] = get_query_var('wdt_md_p_t_col_name');
        $_REQUEST['wdt_md_col_value'] = get_query_var('wdt_md_col_value');
    }
}

// create admin notice
function sfwm_create_notice($msg)
{
    $html = '<div class="updated notice is-dismissible">'
                .'<p>'
                    .$msg.' '
                    .'<a href="https://antero.eu/seo-for-wdt-md/docs.txt">'
                        .'Please see the documentation'
                    .'</a>.'
                .'</p>'
            .'</div>';
    return $html;
}

// show admin notices
function sfwm_admin_notices()
{
    if ( get_transient('sfwm_activation_error_md_old') )
    {
        $msg = '(seo-for-wdt-md) ';
        $msg .= 'NOTE: No required version of wpDataTables Master-Detail!';
        if ( SFWM_CFG["Admin notices"] ) echo sfwm_create_notice($msg);
        delete_transient('sfwm_activation_error_md_old');
    }
    if ( get_transient('sfwm_activation_error_not_configured') )
    {
        $msg = '(seo-for-wdt-md) ';
        $msg .= 'NOTE: The plugin has not been configured yet!';
        if ( SFWM_CFG["Admin notices"] ) echo sfwm_create_notice($msg);
        delete_transient('sfwm_activation_error_not_configured');
    }
    if ( get_transient('sfwm_data_error_id_col') )
    {
        $msg = '(seo-for-wdt-md) ';
        $msg .= 'Could not get ID column for table(s), defaulting to wdt_ID.';
        if ( SFWM_CFG["Admin notices"] ) echo sfwm_create_notice($msg);
        delete_transient('sfwm_data_error_id_col');
    }
    if ( get_transient('sfwm_cfg_error') )
    {
        $msg = '(seo-for-wdt-md) ';
        $msg .= 'Misconfigured value: '.get_transient('sfwm_cfg_error').'.';
        if ( SFWM_CFG["Admin notices"] ) echo sfwm_create_notice($msg);
        delete_transient('sfwm_cfg_error');
    }
}